<?php
// admin/products.php
require_once '../includes/functions.php';

// التحقق من الصلاحيات
if(!isLoggedIn()) redirect('login.php');

 $error = '';
 $success = '';
 $editMode = false;
 $productToEdit = [];
 $galleryImages = [];
 $currentVariants = []; // المتغير الجديد لتخزين التنوعات (المقاس + اللون + الكمية)

// ============================================================
// 1. منطق إدارة الفئات (Categories Logic)
// ============================================================

if(isset($_POST['add_category'])) {
    $cat_name = trim($_POST['cat_name']);
    if(!empty($cat_name)) {
        $stmt = $pdo->prepare("INSERT INTO categories (name) VALUES (?)");
        $stmt->execute([$cat_name]);
        $success = "تم إضافة الفئة بنجاح";
    }
}

if(isset($_POST['update_category'])) {
    $cat_id = $_POST['cat_id'];
    $cat_name = trim($_POST['cat_name']);
    if(!empty($cat_name) && !empty($cat_id)) {
        $stmt = $pdo->prepare("UPDATE categories SET name = ? WHERE id = ?");
        $stmt->execute([$cat_name, $cat_id]);
        $success = "تم تحديث اسم الفئة";
    }
}

if(isset($_GET['delete_cat'])) {
    $cat_id = (int)$_GET['delete_cat'];
    $pdo->prepare("UPDATE products SET category_id = 0 WHERE category_id = ?")->execute([$cat_id]);
    $pdo->prepare("DELETE FROM categories WHERE id = ?")->execute([$cat_id]);
    header("Location: products.php?cat_deleted=1");
    exit;
}

 $categories = [];
try {
    $categories = $pdo->query("SELECT * FROM categories ORDER BY id DESC")->fetchAll();
} catch(Exception $e) { }

// ============================================================
// 2. منطق المنتجات (Products Logic) - مع تعديل نظام المخزون
// ============================================================

if(isset($_GET['delete_gallery_image']) && isset($_GET['edit_id'])) {
    $g_id = (int)$_GET['delete_gallery_image'];
    $p_id = (int)$_GET['edit_id'];
    
    $stmt = $pdo->prepare("SELECT image FROM product_gallery WHERE id = ?");
    $stmt->execute([$g_id]);
    $img = $stmt->fetch();
    
    if($img) {
        $filePath = "../assets/uploads/" . $img['image'];
        if(file_exists($filePath)) unlink($filePath);
        $pdo->prepare("DELETE FROM product_gallery WHERE id = ?")->execute([$g_id]);
        header("Location: products.php?edit=" . $p_id);
        exit;
    }
}

if(isset($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $stmt = $pdo->prepare("SELECT * FROM products WHERE id = ?");
    $stmt->execute([$id]);
    $productToEdit = $stmt->fetch();
    
    if($productToEdit) {
        $editMode = true;
        
        // استرجاع التنوعات (المقاسات المرتبطة بالألوان)
        // سنقوم بتخزين البيانات الجديدة في عمود custom_sizes كون مصفوفة JSON موحدة
        if(!empty($productToEdit['custom_sizes'])) {
            $decoded = json_decode($productToEdit['custom_sizes'], true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                // التحقق من أن البيانات تحتوي على المفتاح الجديد 'color' (للتوافقية)
                if(isset($decoded[0]['color'])) {
                    $currentVariants = $decoded;
                } else {
                    // بيانات قديمة (تحويل تلقائي لعدم ضياع البيانات القديمة)
                    foreach($decoded as $old) {
                        $currentVariants[] = [
                            'size' => $old['size'], 
                            'color' => '#000000', // لون افتراضي
                            'qty' => $old['qty']
                        ];
                    }
                }
            }
        }

        $stmt_g = $pdo->prepare("SELECT * FROM product_gallery WHERE product_id = ?");
        $stmt_g->execute([$id]);
        $galleryImages = $stmt_g->fetchAll();
    }
}

if(isset($_POST['save_product'])) {
    
    $title = trim($_POST['title']);
    $price = $_POST['price'];
    $cat_id = $_POST['category_id'] ?? 0;
    $description = isset($_POST['description']) ? trim($_POST['description']) : '';
    $old_price = !empty($_POST['old_price']) ? $_POST['old_price'] : 0;
    
    // ** معالجة التنوعات الجديدة (المقاس + اللون + الكمية) **
    $variants_data = [];
    if (isset($_POST['variant_size']) && isset($_POST['variant_color']) && isset($_POST['variant_qty'])) {
        for ($i = 0; $i < count($_POST['variant_size']); $i++) {
            $vSize = trim($_POST['variant_size'][$i]);
            $vColor = $_POST['variant_color'][$i];
            $vQty = (int)$_POST['variant_qty'][$i];
            
            if (!empty($vSize) && !empty($vColor)) {
                $variants_data[] = [
                    'size' => $vSize, 
                    'color' => $vColor, 
                    'qty' => $vQty
                ];
            }
        }
    }
    
    // تخزين المصفوفة الموحدة في custom_sizes
    // وتفريغ custom_colors لأننا لم نعد نستخدمه بشكل منفصل
    $inventory_json = json_encode($variants_data, JSON_UNESCAPED_UNICODE);
    $custom_sizes = $inventory_json;
    $custom_colors = ''; // إفراغ الحقل القديم

    $isUpdate = isset($_POST['product_id']) && !empty($_POST['product_id']);
    
    // معالجة الصورة
    $imgName = $isUpdate ? $_POST['old_image'] : 'default.png';
    if(!empty($_FILES['image']['name'])) {
        $fileName = $_FILES['image']['name'];
        $fileTmp  = $_FILES['image']['tmp_name'];
        $fileExt  = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
        if(in_array($fileExt, ['jpg', 'jpeg', 'png', 'webp'])) {
            $newFileName = uniqid('prod_', true) . '.' . $fileExt;
            if(move_uploaded_file($fileTmp, "../assets/uploads/" . $newFileName)) {
                if($isUpdate && $_POST['old_image'] != 'default.png' && file_exists("../assets/uploads/" . $_POST['old_image'])) {
                    unlink("../assets/uploads/" . $_POST['old_image']);
                }
                $imgName = $newFileName;
            }
        }
    }

    // الحفظ في DB
    try {
        if($isUpdate) {
            $id = $_POST['product_id'];
            $sql = "UPDATE products SET title=?, price=?, old_price=?, category_id=?, description=?, image=?, custom_sizes=?, custom_colors=? WHERE id=?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$title, $price, $old_price, $cat_id, $description, $imgName, $custom_sizes, $custom_colors, $id]);
            
            $currentProductId = $id;
            $success = "تم تحديث البيانات والمخزون بنجاح";
            $currentVariants = $variants_data; // تحديث العرض
            $editMode = true;
            // تحديث بيانات العرض
            $productToEdit['title'] = $title;
            $productToEdit['price'] = $price;
            $productToEdit['old_price'] = $old_price;
            $productToEdit['category_id'] = $cat_id; 
            $productToEdit['description'] = $description;
            $productToEdit['image'] = $imgName;

        } else {
            $sql = "INSERT INTO products (title, price, old_price, category_id, description, image, custom_sizes, custom_colors) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$title, $price, $old_price, $cat_id, $description, $imgName, $custom_sizes, $custom_colors]);
            
            $currentProductId = $pdo->lastInsertId();
            $success = "تمت الإضافة بنجاح";
        }

        // الألبوم (Gallery) - نفس الكود السابق
        if(!empty($_FILES['gallery']['name'][0])) {
            $total = count($_FILES['gallery']['name']);
            for($i = 0; $i < $total; $i++) {
                $gName = $_FILES['gallery']['name'][$i];
                $gTmp  = $_FILES['gallery']['tmp_name'][$i];
                $gExt  = strtolower(pathinfo($gName, PATHINFO_EXTENSION));
                if(in_array($gExt, ['jpg', 'jpeg', 'png', 'webp'])) {
                    $newGName = uniqid('gal_', true) . '.' . $gExt;
                    if(move_uploaded_file($gTmp, "../assets/uploads/" . $newGName)) {
                        $stmtG = $pdo->prepare("INSERT INTO product_gallery (product_id, image) VALUES (?, ?)");
                        $stmtG->execute([$currentProductId, $newGName]);
                    }
                }
            }
            // تحديث قائمة الصور للعرض
            $stmt_g = $pdo->prepare("SELECT * FROM product_gallery WHERE product_id = ?");
            $stmt_g->execute([$currentProductId]);
            $galleryImages = $stmt_g->fetchAll();
        }

    } catch(PDOException $e) {
        $error = "خطأ: " . $e->getMessage();
    }
}

if(isset($_GET['delete'])) {
    $id = $_GET['delete'];
    $stmt = $pdo->prepare("SELECT image FROM products WHERE id = ?");
    $stmt->execute([$id]);
    $prod = $stmt->fetch();
    
    if($prod && $prod['image'] != 'default.png' && file_exists("../assets/uploads/" . $prod['image'])) unlink("../assets/uploads/" . $prod['image']);

    $stmtG = $pdo->prepare("SELECT image FROM product_gallery WHERE product_id = ?");
    $stmtG->execute([$id]);
    $gImgs = $stmtG->fetchAll();
    foreach($gImgs as $g) {
        if(file_exists("../assets/uploads/" . $g['image'])) unlink("../assets/uploads/" . $g['image']);
    }
    
    $pdo->prepare("DELETE FROM product_gallery WHERE product_id = ?")->execute([$id]);
    $pdo->prepare("DELETE FROM products WHERE id = ?")->execute([$id]);
    redirect('products.php');
}

 $sqlProducts = "SELECT products.*, categories.name as category_name 
                FROM products 
                LEFT JOIN categories ON products.category_id = categories.id 
                ORDER BY products.id DESC";
 $products = $pdo->query($sqlProducts)->fetchAll();
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <title>إدارة المنتجات والفئات</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700;800&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Cairo', sans-serif; }
        [x-cloak] { display: none !important; }
        .fade-in { animation: fadeIn 0.3s ease-out; }
        @keyframes fadeIn { from { opacity: 0; transform: scale(0.95); } to { opacity: 1; transform: scale(1); } }
        
        /* تحسينات التصميم الاحترافي */
        .glass-nav { background: rgba(255, 255, 255, 0.05); }
    </style>
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.13.3/dist/cdn.min.js"></script>
</head>
<body class="bg-slate-50 text-slate-800" x-data="{ showCatModal: false, catEditMode: false, catName: '', catId: '' }">

<div class="flex h-screen overflow-hidden">
    
    <!-- Sidebar (تصميم أغمق وأكثر أناقة) -->
    <aside class="w-64 bg-slate-900 text-white hidden md:flex flex-col shadow-2xl z-20">
        <div class="h-24 flex items-center justify-center border-b border-slate-800 bg-gradient-to-b from-slate-900 to-slate-800">
            <div class="flex flex-col items-center">
                <h2 class="text-2xl font-bold text-emerald-400 tracking-wide">لوحة التحكم</h2>
                <span class="text-[10px] text-slate-500 uppercase tracking-widest">Admin Panel</span>
            </div>
        </div>
        
        <!-- Menu Items -->
        <nav class="flex-1 px-4 py-8 space-y-2 overflow-y-auto">
            
            <!-- الرئيسية (تم تصحيح الأيقونة والرابط) -->
            <a href="index.php" class="flex items-center gap-4 px-4 py-3 text-slate-400 hover:bg-slate-800 hover:text-white rounded-2xl transition-all duration-300 group">
                <svg class="w-6 h-6 group-hover:text-emerald-400 transition-colors" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 21v-4a2 2 0 012-2h4a2 2 0 012 2v4" />
                </svg>
                <span class="font-bold text-sm">الرئيسية</span>
            </a>

            <!-- المنتجات -->
            <a href="products.php" class="flex items-center gap-4 px-4 py-3 bg-emerald-600 text-white rounded-2xl shadow-lg shadow-emerald-900/20 transition-all duration-300 font-bold border border-emerald-500/30">
                <svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"/>
                </svg>
                المنتجات
            </a>

            <!-- الإعدادات (تم تصحيح الأيقونة) -->
            <a href="settings.php" class="flex items-center gap-4 px-4 py-3 text-slate-400 hover:bg-slate-800 hover:text-white rounded-2xl transition-all duration-300 group">
                <svg class="w-6 h-6 group-hover:text-blue-400 transition-colors" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                </svg>
                الإعدادات
            </a>

            <!-- الإضافات (تم تصحيح الأيقونة) -->
            <a href="plugins.php" class="flex items-center gap-4 px-4 py-3 text-slate-400 hover:bg-slate-800 hover:text-white rounded-2xl transition-all duration-300 group">
                <svg class="w-6 h-6 group-hover:text-purple-400 transition-colors" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4" />
                </svg>
                الإضافات
            </a>
        </nav>

        <div class="p-6 border-t border-slate-800">
            <a href="logout.php" class="flex items-center gap-2 text-red-400 hover:text-red-300 font-bold text-sm transition-colors">تسجيل خروج</a>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="flex-1 overflow-y-auto relative">
        <!-- Mobile Header -->
        <div class="md:hidden bg-slate-900 text-white p-4 flex justify-between items-center shadow-md">
            <span class="font-bold text-lg">لوحة التحكم</span>
            <a href="logout.php" class="text-red-400 text-sm bg-red-900/20 px-3 py-1 rounded-lg">خروج</a>
        </div>

        <div class="p-8 lg:p-12 max-w-[1600px] mx-auto">
            
            <!-- Header Actions -->
            <div class="flex flex-col lg:flex-row justify-between items-center mb-10 gap-6 bg-white p-8 rounded-3xl shadow-sm border border-slate-200">
                <div>
                    <h1 class="text-3xl lg:text-4xl font-black text-slate-800 tracking-tight">إدارة المنتجات</h1>
                    <p class="text-slate-500 text-sm mt-2">قم بإدارة المنتجات، المخزون، وتصنيفات متجرك بكل سهولة واحترافية.</p>
                </div>
                <div class="flex gap-4">
                    <button @click="showCatModal = true" class="bg-slate-100 hover:bg-slate-200 text-slate-700 px-6 py-3 rounded-2xl font-bold shadow-sm transition flex items-center gap-3">
                        <svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h7"/></svg>
                        إدارة الفئات
                    </button>
                    <?php if(!$editMode): ?>
                    <button onclick="document.getElementById('formCard').classList.toggle('hidden')" 
                            class="bg-emerald-600 hover:bg-emerald-700 text-white px-6 py-3 rounded-2xl font-bold shadow-lg shadow-emerald-500/30 transition flex items-center gap-3 active:scale-95 duration-200">
                        <svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                        منتج جديد
                    </button>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Alerts -->
            <?php if($error): ?>
                <div class="bg-red-50 text-red-700 p-6 rounded-2xl border border-red-200 mb-8 font-bold flex items-center gap-4 animate-fade-in shadow-sm">
                    <svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" /></svg>
                    <?= $error ?>
                </div>
            <?php endif; ?>
            <?php if($success): ?>
                <div class="bg-emerald-50 text-emerald-700 p-6 rounded-2xl border border-emerald-200 mb-8 font-bold flex items-center gap-4 animate-fade-in shadow-sm">
                    <svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" /></svg>
                    <?= $success ?>
                </div>
            <?php endif; ?>

            <!-- Form Card -->
            <div id="formCard" class="<?= $editMode ? '' : 'hidden' ?> bg-white rounded-3xl shadow-xl border border-slate-200 overflow-hidden mb-12 animate-fade-in">
                <div class="flex justify-between items-center px-8 py-6 border-b border-slate-100 bg-slate-50/50">
                    <div class="flex items-center gap-3">
                        <div class="p-2 bg-indigo-100 rounded-xl text-indigo-600">
                            <svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/></svg>
                        </div>
                        <h3 class="text-xl font-bold text-slate-800"><?= $editMode ? 'تعديل بيانات المنتج' : 'إضافة منتج جديد' ?></h3>
                    </div>
                    <?php if($editMode): ?>
                        <a href="products.php" class="text-sm text-slate-500 hover:text-red-500 font-semibold transition-colors border border-slate-200 hover:border-red-200 px-4 py-2 rounded-lg">إلغاء التعديل</a>
                    <?php endif; ?>
                </div>

                <form method="POST" enctype="multipart/form-data" class="grid grid-cols-1 lg:grid-cols-12 gap-8 p-8">
                    <?php if($editMode): ?>
                        <input type="hidden" name="product_id" value="<?= $productToEdit['id'] ?>">
                        <input type="hidden" name="old_image" value="<?= $productToEdit['image'] ?>">
                    <?php endif; ?>

                    <div class="lg:col-span-8 space-y-8">
                        <!-- Basic Info -->
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div class="col-span-2">
                                <label class="block text-sm font-bold text-slate-700 mb-3">اسم المنتج</label>
                                <input type="text" name="title" required value="<?= $editMode ? htmlspecialchars($productToEdit['title']) : '' ?>" 
                                       class="w-full px-5 py-3 rounded-xl border border-slate-300 focus:border-indigo-500 focus:ring-4 focus:ring-indigo-100 outline-none transition-all duration-200 bg-slate-50 focus:bg-white text-slate-800 font-medium">
                            </div>
                            <div>
                                <label class="block text-sm font-bold text-slate-700 mb-3">السعر الحالي</label>
                                <div class="relative">
                                    <input type="number" step="0.01" name="price" required value="<?= $editMode ? $productToEdit['price'] : '' ?>" 
                                           class="w-full pl-12 pr-4 py-3 rounded-xl border border-slate-300 focus:border-emerald-500 focus:ring-4 focus:ring-emerald-100 outline-none transition-all duration-200 bg-slate-50 focus:bg-white text-slate-800 font-bold">
                                    <span class="absolute left-4 top-1/2 -translate-y-1/2 text-slate-400">ج.م</span>
                                </div>
                            </div>
                            <div>
                                <label class="block text-sm font-bold text-red-600 mb-3">السعر قبل الخصم</label>
                                <input type="number" step="0.01" name="old_price" value="<?= ($editMode && isset($productToEdit['old_price'])) ? $productToEdit['old_price'] : '' ?>" 
                                       class="w-full px-5 py-3 rounded-xl border border-red-200 focus:border-red-500 focus:ring-4 focus:ring-red-100 outline-none transition-all duration-200 bg-red-50/30 focus:bg-red-50/50 text-red-900 font-medium">
                            </div>
                        </div>

                        <!-- Category Selection -->
                        <div class="bg-slate-50 p-6 rounded-2xl border border-slate-200">
                            <div class="flex justify-between items-center mb-3">
                                <label class="block text-sm font-bold text-slate-700">القسم / الفئة</label>
                                <button type="button" @click="showCatModal = true" class="text-xs font-bold text-indigo-600 hover:text-indigo-700 hover:underline">+ إدارة الفئات</button>
                            </div>
                            <select name="category_id" class="w-full px-5 py-3 rounded-xl border border-slate-300 focus:border-indigo-500 focus:ring-4 focus:ring-indigo-100 outline-none transition-all duration-200 bg-white text-slate-700 font-medium cursor-pointer shadow-sm">
                                <option value="0">بدون قسم</option>
                                <?php foreach($categories as $cat): ?>
                                    <option value="<?= $cat['id'] ?>" <?= ($editMode && isset($productToEdit['category_id']) && $productToEdit['category_id'] == $cat['id']) ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($cat['name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Inventory Matrix (New Section) -->
                        <div class="bg-indigo-50/30 p-8 rounded-3xl border border-indigo-100 shadow-sm relative overflow-hidden">
                            <div class="absolute top-0 right-0 w-32 h-32 bg-indigo-500/10 rounded-full -mr-16 -mt-16 blur-2xl"></div>
                            
                            <label class="block text-sm font-bold text-indigo-800 mb-6 flex items-center gap-3">
                                <svg class="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"/></svg>
                                المخزون (المقاس + اللون + الكمية)
                            </label>
                            
                            <div class="bg-white p-5 rounded-2xl shadow-sm border border-indigo-200 mb-6">
                                <div class="flex flex-col md:flex-row gap-4 items-end">
                                    <div class="w-full md:w-1/3">
                                        <label class="block text-xs font-bold text-slate-500 mb-2">المقاس</label>
                                        <input type="text" id="varSize" class="w-full px-4 py-2.5 rounded-xl border border-slate-300 focus:border-indigo-500 focus:ring-2 focus:ring-indigo-200 outline-none transition-all duration-200 bg-slate-50 text-sm font-bold" placeholder="مثال: XL">
                                    </div>
                                    <div class="w-full md:w-1/3">
                                        <label class="block text-xs font-bold text-slate-500 mb-2">اللون</label>
                                        <div class="flex items-center gap-3 border border-slate-300 rounded-xl px-3 py-2 bg-slate-50">
                                            <input type="color" id="varColor" class="w-8 h-8 rounded cursor-pointer border-0 bg-transparent">
                                            <span class="text-xs text-slate-400 font-mono" id="colorHexDisplay">#000000</span>
                                        </div>
                                    </div>
                                    <div class="w-full md:w-1/3">
                                        <label class="block text-xs font-bold text-slate-500 mb-2">الكمية المتاحة</label>
                                        <input type="number" id="varQty" class="w-full px-4 py-2.5 rounded-xl border border-slate-300 focus:border-indigo-500 focus:ring-2 focus:ring-indigo-200 outline-none transition-all duration-200 bg-slate-50 text-sm font-bold" placeholder="مثال: 10">
                                    </div>
                                    <div class="w-full md:w-auto">
                                        <button type="button" onclick="addVariantRow()" class="w-full bg-indigo-600 hover:bg-indigo-700 text-white py-2.5 rounded-xl font-bold text-sm shadow-md shadow-indigo-500/20 transition-all duration-200 flex items-center justify-center gap-2">
                                            <svg class="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                                            إضافة للقائمة +
                                        </button>
                                    </div>
                                </div>
                            </div>

                            <div id="variantsList" class="space-y-3">
                                <?php if($editMode && !empty($currentVariants)): ?>
                                    <?php foreach($currentVariants as $v): ?>
                                    <div class="flex items-center gap-4 bg-white p-3 rounded-2xl border border-slate-200 shadow-sm hover:shadow-md transition-all duration-200 group relative">
                                        <div class="w-10 h-10 rounded-full border-2 border-slate-100 shadow-inner flex items-center justify-center" style="background-color: <?= $v['color'] ?>;"></div>
                                        
                                        <input type="hidden" name="variant_color[]" value="<?= $v['color'] ?>">
                                        <input type="hidden" name="variant_size[]" value="<?= htmlspecialchars($v['size']) ?>">

                                        <div class="flex flex-col flex-1 min-w-[100px]">
                                            <label class="text-[10px] text-slate-400 font-bold uppercase tracking-wider">مقاس</label>
                                            <span class="text-sm font-bold text-slate-800 bg-slate-100 px-3 py-1 rounded-lg inline-block w-full text-center"><?= htmlspecialchars($v['size']) ?></span>
                                        </div>

                                        <div class="h-8 w-px bg-slate-200 mx-2"></div>

                                        <div class="flex flex-col flex-1">
                                            <label class="text-[10px] text-slate-400 font-bold uppercase tracking-wider">كمية</label>
                                            <input type="number" name="variant_qty[]" value="<?= (int)$v['qty'] ?>" class="w-20 px-2 py-1.5 border border-slate-300 rounded-lg text-sm font-bold text-slate-700 focus:border-indigo-500 focus:ring-1 focus:ring-indigo-200 outline-none transition-all duration-200">
                                        </div>

                                        <button type="button" onclick="this.parentElement.remove()" class="absolute -top-2 -left-2 bg-red-50 text-red-500 hover:bg-red-500 hover:text-white rounded-full p-1.5 transition-all duration-200 shadow-sm" title="حذف">
                                            <svg class="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
                                        </button>
                                    </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Description -->
                        <div>
                            <label class="block text-sm font-bold text-slate-700 mb-3">وصف المنتج</label>
                            <textarea name="description" rows="6" class="w-full px-5 py-3 rounded-xl border border-slate-300 focus:border-indigo-500 focus:ring-4 focus:ring-indigo-100 outline-none transition-all duration-200 bg-slate-50 focus:bg-white text-slate-700 leading-relaxed resize-none font-medium"><?= $editMode ? htmlspecialchars($productToEdit['description']) : '' ?></textarea>
                        </div>
                    </div>

                    <!-- Right Column: Images -->
                    <div class="lg:col-span-4 space-y-6">
                        <div class="bg-slate-50 p-6 rounded-3xl border border-slate-200 text-center">
                            <label class="block text-sm font-bold text-slate-700 mb-4">الصورة الرئيسية</label>
                            <?php if($editMode): ?>
                                <div class="relative w-full aspect-square rounded-2xl overflow-hidden shadow-md bg-white mb-4">
                                    <img src="../assets/uploads/<?= $productToEdit['image'] ?>" class="w-full h-full object-contain">
                                    <span class="absolute bottom-2 left-2 bg-black/50 text-white text-[10px] px-2 py-1 rounded-md">الصورة الحالية</span>
                                </div>
                            <?php endif; ?>
                            <label class="w-full py-10 border-2 border-dashed border-slate-300 rounded-2xl cursor-pointer hover:border-indigo-400 hover:bg-indigo-50/30 transition-all duration-300 flex flex-col items-center justify-center relative group">
                                <input type="file" name="image" accept="image/*" class="absolute inset-0 w-full h-full opacity-0 cursor-pointer">
                                <div class="w-12 h-12 bg-slate-200 rounded-full flex items-center justify-center mb-3 group-hover:bg-indigo-100 group-hover:text-indigo-600 transition-colors">
                                    <svg class="w-6 h-6 text-slate-500 group-hover:text-indigo-600" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path></svg>
                                </div>
                                <span class="text-xs font-semibold text-slate-500 group-hover:text-indigo-600 transition-colors">اضغط لرفع صورة</span>
                                <span class="text-[10px] text-slate-400">JPG, PNG, WEBP</span>
                            </label>
                        </div>

                        <div class="bg-blue-50 p-6 rounded-3xl border border-blue-200 text-center">
                            <label class="block text-sm font-bold text-blue-800 mb-4">ألبوم الصور (متعددة)</label>
                            <label class="w-full py-8 border-2 border-dashed border-blue-300 rounded-2xl cursor-pointer hover:border-blue-400 hover:bg-blue-100/30 transition-all duration-300 flex flex-col items-center justify-center relative group">
                                <input type="file" name="gallery[]" multiple accept="image/*" class="absolute inset-0 w-full h-full opacity-0 cursor-pointer">
                                <div class="w-12 h-12 bg-blue-200 rounded-full flex items-center justify-center mb-3 group-hover:bg-blue-100 group-hover:text-blue-600 transition-colors">
                                    <svg class="w-6 h-6 text-blue-500 group-hover:text-blue-600" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path></svg>
                                </div>
                                <span class="text-xs font-semibold text-blue-600 group-hover:text-blue-700 transition-colors">اضغط لرفع صور متعددة</span>
                            </label>
                        </div>

                        <?php if($editMode && !empty($galleryImages)): ?>
                            <div class="border-t pt-6">
                                <label class="block text-xs font-bold text-slate-500 mb-4 uppercase tracking-wide">صور الألبوم الحالية</label>
                                <div class="grid grid-cols-3 gap-3">
                                    <?php foreach($galleryImages as $gImg): ?>
                                    <div class="relative group aspect-square">
                                        <img src="../assets/uploads/<?= $gImg['image'] ?>" class="w-full h-full object-cover rounded-xl shadow-sm">
                                        <a href="products.php?edit_id=<?= $productToEdit['id'] ?>&delete_gallery_image=<?= $gImg['id'] ?>" 
                                           onclick="return confirm('حذف هذه الصورة؟')"
                                           class="absolute top-2 left-2 bg-red-500 hover:bg-red-600 text-white w-6 h-6 rounded-lg flex items-center justify-center text-xs shadow-lg opacity-0 group-hover:opacity-100 transition-all duration-200">
                                            <svg class="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" /></svg>
                                        </a>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>

                    <div class="lg:col-span-12 pt-6 border-t border-slate-100 mt-6">
                        <button type="submit" name="save_product" class="w-full bg-emerald-600 hover:bg-emerald-700 text-white py-4 rounded-2xl font-bold text-lg shadow-xl shadow-emerald-600/30 transition-all duration-300 transform active:scale-[0.99] flex items-center justify-center gap-3">
                            <?= $editMode ? 'حفظ التعديلات (وتحديث المخزون)' : 'إضافة المنتج' ?>
                            <svg class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7H5a2 2 0 00-2 2v9a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-3m-1 4l-3 3m0 0l-3-3m3 3V4" /></svg>
                        </button>
                    </div>
                </form>
            </div>

            <!-- Products Table -->
            <div class="bg-white rounded-3xl shadow-xl border border-slate-200 overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="w-full text-right">
                        <thead class="bg-slate-50 border-b border-slate-200">
                            <tr>
                                <th class="p-6 text-sm font-bold text-slate-600 uppercase tracking-wider">الصورة</th>
                                <th class="p-6 text-sm font-bold text-slate-600 uppercase tracking-wider">التفاصيل</th>
                                <th class="p-6 text-sm font-bold text-slate-600 uppercase tracking-wider">الفئة</th>
                                <th class="p-6 text-sm font-bold text-slate-600 uppercase tracking-wider">المخزون (متوفر)</th>
                                <th class="p-6 text-sm font-bold text-slate-600 uppercase tracking-wider">السعر</th>
                                <th class="p-6 text-sm font-bold text-slate-600 uppercase tracking-wider text-center">إجراءات</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-slate-100">
                            <?php foreach($products as $p): ?>
                            <tr class="hover:bg-slate-50 transition duration-200 group">
                                <td class="p-4 w-24 relative">
                                    <?php if(isset($p['old_price']) && $p['old_price'] > $p['price']): ?>
                                        <span class="absolute top-2 left-2 z-10 bg-red-500 text-white text-[10px] px-2 py-1 rounded-lg font-bold shadow-md animate-pulse">خصم</span>
                                    <?php endif; ?>
                                    <div class="w-16 h-16 rounded-xl border border-slate-200 shadow-sm bg-white p-1">
                                        <img src="../assets/uploads/<?= !empty($p['image']) ? $p['image'] : 'default.png' ?>" class="w-full h-full object-cover rounded-lg">
                                    </div>
                                </td>
                                
                                <td class="p-4 align-top">
                                    <div class="font-bold text-slate-800 text-lg mb-1"><?= $p['title'] ?></div>
                                    <div class="text-xs text-slate-400 font-normal line-clamp-2 h-8 overflow-hidden"><?= mb_substr($p['description'], 0, 60) ?>...</div>
                                </td>

                                <td class="p-4 align-top">
                                    <?php if($p['category_name']): ?>
                                        <span class="bg-slate-100 text-slate-600 px-4 py-2 rounded-full text-xs font-bold border border-slate-200 inline-block"><?= htmlspecialchars($p['category_name']) ?></span>
                                    <?php else: ?>
                                        <span class="text-xs text-slate-300">---</span>
                                    <?php endif; ?>
                                </td>

                                <!-- Inventory Display -->
                                <td class="p-4 align-top min-w-[200px]">
                                    <?php 
                                    // نقرأ البيانات من custom_sizes
                                    if(!empty($p['custom_sizes'])): 
                                        $variants = json_decode($p['custom_sizes'], true);
                                        if(is_array($variants) && count($variants) > 0): ?>
                                            <div class="flex flex-wrap gap-2">
                                                <?php foreach($variants as $v): ?>
                                                    <div class="flex items-center gap-2 bg-white border border-slate-200 rounded-full px-2 py-1 pr-3 shadow-sm">
                                                        <div class="w-3 h-3 rounded-full border border-slate-300 shadow-sm" style="background-color: <?= $v['color'] ?>;"></div>
                                                        <span class="text-xs font-bold text-slate-700"><?= $v['size'] ?></span>
                                                        <span class="text-[10px] font-bold px-1.5 rounded <?= (int)$v['qty'] > 0 ? 'bg-emerald-100 text-emerald-700' : 'bg-red-100 text-red-700' ?>">(<?= $v['qty'] ?>)</span>
                                                    </div>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php else: ?>
                                            <span class="text-xs text-slate-400">لا توجد تنوعات</span>
                                        <?php endif; 
                                    else: ?>
                                        <span class="text-xs text-slate-300">---</span>
                                    <?php endif; ?>
                                </td>

                                <td class="p-4 align-top">
                                    <div class="flex flex-col">
                                        <span class="font-black text-emerald-600 text-xl"><?= number_format($p['price'], 2) ?></span>
                                        <?php if(isset($p['old_price']) && $p['old_price'] > $p['price']): ?>
                                            <span class="text-sm text-red-400 line-through font-bold decoration-2 opacity-70"><?= number_format($p['old_price'], 2) ?></span>
                                        <?php endif; ?>
                                    </div>
                                </td>

                                <td class="p-4 text-center align-middle">
                                    <div class="flex items-center justify-center gap-2">
                                        <a href="?edit=<?= $p['id'] ?>" class="bg-blue-50 text-blue-600 p-2.5 rounded-xl hover:bg-blue-600 hover:text-white transition-all duration-200 shadow-sm group/btn" title="تعديل">
                                            <svg class="w-5 h-5 group-hover/btn:scale-110 transition-transform" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/></svg>
                                        </a>
                                        <a href="?delete=<?= $p['id'] ?>" onclick="return confirm('هل أنت متأكد من الحذف؟')" class="bg-red-50 text-red-500 p-2.5 rounded-xl hover:bg-red-500 hover:text-white transition-all duration-200 shadow-sm group/btn" title="حذف">
                                            <svg class="w-5 h-5 group-hover/btn:scale-110 transition-transform" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>

    <!-- Modal للفئات -->
    <div x-show="showCatModal" x-cloak class="relative z-50" aria-labelledby="modal-title" role="dialog" aria-modal="true">
        <div x-show="showCatModal" x-transition.opacity class="fixed inset-0 bg-slate-900/60 backdrop-blur-sm" @click="showCatModal = false"></div>

        <div class="fixed inset-0 z-10 overflow-y-auto">
            <div class="flex min-h-full items-center justify-center p-4 text-center sm:p-0">
                <div x-show="showCatModal" 
                     x-transition:enter="ease-out duration-300" 
                     x-transition:enter-start="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95" 
                     x-transition:enter-end="opacity-100 translate-y-0 sm:scale-100"
                     class="relative transform overflow-hidden rounded-3xl bg-white text-left shadow-2xl transition-all sm:my-8 sm:w-full sm:max-w-lg">
                    
                    <div class="bg-white px-4 pb-4 pt-6 sm:p-6 sm:pb-4">
                        <div class="text-right">
                            <h3 class="text-2xl font-bold leading-6 text-slate-800 mb-6" id="modal-title">إدارة الفئات (الأقسام)</h3>
                            
                            <form method="POST" class="mb-8 bg-slate-50 p-6 rounded-2xl border border-slate-200">
                                <label class="block text-sm font-bold text-slate-700 mb-3" x-text="catEditMode ? 'تعديل اسم الفئة' : 'إضافة فئة جديدة'"></label>
                                <div class="flex gap-3">
                                    <input type="text" name="cat_name" x-model="catName" required class="flex-1 px-4 py-3 border rounded-xl focus:ring-2 focus:ring-indigo-500 outline-none text-slate-800 bg-white shadow-sm" placeholder="اكتب اسم الفئة هنا...">
                                    <input type="hidden" name="cat_id" x-model="catId">
                                    
                                    <button type="submit" :name="catEditMode ? 'update_category' : 'add_category'" 
                                            class="px-6 py-3 text-white rounded-xl font-bold transition-all duration-200 shadow-md flex-1 justify-center items-center gap-2"
                                            :class="catEditMode ? 'bg-blue-600 hover:bg-blue-700' : 'bg-indigo-600 hover:bg-indigo-700'">
                                        <span x-text="catEditMode ? 'تحديث' : 'إضافة'"></span>
                                        <svg class="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/></svg>
                                    </button>
                                    
                                    <button type="button" x-show="catEditMode" @click="catEditMode = false; catName = ''; catId = ''" class="px-4 py-3 bg-slate-200 hover:bg-slate-300 text-slate-600 rounded-xl transition-colors font-bold">إلغاء</button>
                                </div>
                            </form>

                            <div class="max-h-60 overflow-y-auto space-y-3 pr-1">
                                <?php if(empty($categories)): ?>
                                    <p class="text-center text-slate-400 py-8 bg-slate-50 rounded-2xl border border-dashed border-slate-200">لا توجد فئات حالياً</p>
                                <?php else: ?>
                                    <?php foreach($categories as $cat): ?>
                                    <div class="flex justify-between items-center bg-white p-4 border border-slate-200 rounded-xl hover:border-indigo-300 hover:shadow-sm transition-all duration-200">
                                        <span class="font-bold text-slate-800 text-lg"><?= htmlspecialchars($cat['name']) ?></span>
                                        <div class="flex gap-2">
                                            <button @click="catEditMode = true; catName = '<?= htmlspecialchars($cat['name']) ?>'; catId = '<?= $cat['id'] ?>'" class="text-blue-500 hover:text-blue-600 hover:bg-blue-50 p-2 rounded-lg transition-colors">
                                                <svg class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/></svg>
                                            </button>
                                            <a href="products.php?delete_cat=<?= $cat['id'] ?>" onclick="return confirm('حذف الفئة؟ سيتم إلغاء ربط المنتجات التابعة لها.')" class="text-red-500 hover:text-red-600 hover:bg-red-50 p-2 rounded-lg transition-colors">
                                                <svg class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
                                            </a>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <div class="bg-slate-50 px-6 py-4 sm:flex sm:flex-row-reverse sm:px-6">
                        <button type="button" @click="showCatModal = false" class="mt-3 sm:mt-0 inline-flex w-full justify-center rounded-xl bg-white px-5 py-3 text-sm font-semibold text-slate-900 shadow-sm ring-1 ring-inset ring-slate-300 hover:bg-slate-50 sm:w-auto transition-all">إغلاق</button>
                    </div>
                </div>
            </div>
        </div>
    </div>

</div>

<script>
// دالة إضافة صف "تنوع" جديد (مقاس + لون + كمية)
function addVariantRow() {
    const sizeInput = document.getElementById('varSize');
    const colorInput = document.getElementById('varColor');
    const qtyInput = document.getElementById('varQty');
    const container = document.getElementById('variantsList');
    
    const sizeVal = sizeInput.value.trim();
    const colorVal = colorInput.value;
    const qtyVal = parseInt(qtyInput.value);

    if(!sizeVal || isNaN(qtyVal)) {
        alert('الرجاء كتابة المقاس والكمية بشكل صحيح');
        return;
    }

    // تحديث عرض كود اللون عند التغيير
    document.getElementById('colorHexDisplay').innerText = colorVal;

    const div = document.createElement('div');
    div.className = 'flex items-center gap-4 bg-white p-3 rounded-2xl border border-slate-200 shadow-sm hover:shadow-md transition-all duration-200 group relative';
    
    div.innerHTML = `
        <div class="w-10 h-10 rounded-full border-2 border-slate-100 shadow-inner flex items-center justify-center shrink-0" style="background-color: ${colorVal};"></div>
        <input type="hidden" name="variant_size[]" value="${sizeVal}">
        <input type="hidden" name="variant_color[]" value="${colorVal}">
        
        <div class="flex flex-col flex-1 min-w-[100px]">
            <label class="text-[10px] text-slate-400 font-bold uppercase tracking-wider">مقاس</label>
            <span class="text-sm font-bold text-slate-800 bg-slate-100 px-3 py-1 rounded-lg inline-block w-full text-center shadow-sm">${sizeVal}</span>
        </div>

        <div class="h-8 w-px bg-slate-200 mx-2"></div>

        <div class="flex flex-col flex-1">
            <label class="text-[10px] text-slate-400 font-bold uppercase tracking-wider">كمية</label>
            <input type="number" name="variant_qty[]" value="${qtyVal}" class="w-20 px-2 py-1.5 border border-slate-300 rounded-lg text-sm font-bold text-slate-700 focus:border-indigo-500 focus:ring-1 focus:ring-indigo-200 outline-none transition-all duration-200">
        </div>

        <button type="button" onclick="this.parentElement.remove()" class="absolute -top-2 -left-2 bg-red-50 text-red-500 hover:bg-red-500 hover:text-white rounded-full p-1.5 transition-all duration-200 shadow-sm" title="حذف">
            <svg class="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
        </button>
    `;

    container.appendChild(div);
    sizeInput.value = '';
    qtyInput.value = '';
    sizeInput.focus();
}

// Event Listener للون لعرض الكود
document.getElementById('varColor').addEventListener('input', function(e) {
    document.getElementById('colorHexDisplay').innerText = e.target.value.toUpperCase();
});
</script>

</body>
</html>